import React, { useEffect, useState } from 'react';
import "../styles/Dashboard.css";


const Dashboard = ({ onLogout }) => {
  const [units, setUnits] = useState([]);
  const [englishName, setEnglishName] = useState("");
  const [loading, setLoading] = useState(true);
  const [expandedIdx, setExpandedIdx] = useState(null);
  const [installmentsMap, setInstallmentsMap] = useState({});
  const [installmentsLoading, setInstallmentsLoading] = useState(false);
  const [activeTab, setActiveTab] = useState("Paid");


  const fetchInstallments = async (unit, idx) => {
    try {
      const res = await fetch(`/api/installments/${unit.contract}`, {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('token')}`,
        },
      });
      const data = await res.json();
      return { idx, installments: data.installments || [] };
    } catch (error) {
      console.error('Error fetching installments:', error);
      return { idx, installments: [] };
    }
  };

  useEffect(() => {
    const fetchData = async () => {
      try {
        const res = await fetch('/api/units', {
          headers: {
            'Authorization': `Bearer ${localStorage.getItem('token')}`,
          },
        });
        const data = await res.json();
        setUnits(data.units || []);
        setEnglishName(data.englishName || "");

        // Fetch installments for all units
        const installmentPromises = data.units.map((unit, idx) => 
          fetchInstallments(unit, idx)
        );
        const installmentResults = await Promise.all(installmentPromises);
        
        const newInstallmentsMap = installmentResults.reduce((acc, { idx, installments }) => {
          acc[idx] = installments;
          return acc;
        }, {});
        
        setInstallmentsMap(newInstallmentsMap);
        setLoading(false);
      } catch (error) {
        console.error('Error:', error);
        setLoading(false);
      }
    };

    fetchData();
  }, []);

  const handleViewInstallments = (unit, idx) => {
    setExpandedIdx(expandedIdx === idx ? null : idx);
  };

const handlePayNow = async (inst, unit) => {
  try {
    const orderId = `MIP-${unit.contract}-${Date.now()}`; // unique merchant_reference


    const projectName = (unit.project || unit.projectName || "").trim().toUpperCase();


    const resp = await fetch('/api/pay', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${localStorage.getItem('token')}`
      },
      body: JSON.stringify({
        project: projectName,
        // project: unit.project || unit.projectName || "",
        amount: inst.amount,
        currency: 'EGP',
        orderId,
        description: `Installment payment for ${unit.unit}`,
        customerEmail: unit.email, // optional but good
        customerPhone: unit.customerPhone || '201001234567', // optional fallback
      })
    });

    const data = await resp.json();

    if (!resp.ok) {
      console.error('❌ Payment init failed', data);
      alert('Payment initiation failed: ' + (data.message || JSON.stringify(data)));
      return;
    }

    // PayFort (Payment Link API) response
    const payResp = data.payResponse || {};
    const redirectUrl =
      data.redirectUrl || // returned by backend
      payResp.payment_link || // preferred for PAYMENT_LINK flow
      payResp.hosted_payment_url ||
      payResp.redirect_url ||
      payResp.redirectUrl ||
      payResp['3ds_url'] ||
      (payResp.sdk_token && `https://sbpaymentservices.payfort.com/??token=${payResp.sdk_token}`) ||
      null;

    if (redirectUrl) {
      // console.log('🔗 Redirecting to payment page:', redirectUrl);
      //window.location.href = redirectUrl; // go to payment page
      window.open(redirectUrl, "_blank");

    } else {
      console.warn('⚠️ No redirect URL found. Full response:', payResp);
      alert('No payment link returned. Please try again later.');
    }
  } catch (err) {
    console.error('❌ handlePayNow error', err);
    alert('Payment error: ' + err.message);
  }
};

  // Calculate summary statistics
  const totalUnits = units.length;
  const totalAmount = units.reduce((sum, u) => sum + Number(u.soldprice || 0), 0);
  const allInstallments = Object.values(installmentsMap).flat();
  const paidAmount = allInstallments
    .filter(inst => inst.status === "Paid")
    .reduce((sum, inst) => sum + Number(inst.amount || 0), 0);
  const completion = totalAmount > 0 ? Math.round((paidAmount / totalAmount) * 100) : 0;

  if (loading) return <div>Loading units...</div>;

  return (
    <div className="dashboard-container">
  <button className="logout-button" onClick={onLogout}>
    <span>Logout</span>
    <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24"
         strokeWidth="1.5" stroke="currentColor" className="logout-icon">
      <path strokeLinecap="round" strokeLinejoin="round"
        d="M15.75 9V5.25A2.25 2.25 0 0013.5 3h-6A2.25 2.25 0 005.25 5.25v13.5A2.25 2.25 0 007.5 21h6a2.25 2.25 0 002.25-2.25V15M18 12H9m9 0l-3-3m3 3l-3 3"/>
    </svg>
  </button>


      <div className="company-logo">
        <img src="/MIPLOGO.png" alt="Misr Italia Properties" />
      </div>
      {/* Summary Section */}
      <div className="dashboard-summary">
        <div className="summary-info">
          <h3 className="dashboard-greeting">Hello, {englishName}</h3>
          <div className="summary-cards">
            <div className="summary-card">
              <span className="summary-icon"><i className="fas fa-building"></i></span>
              <div>Total Units</div>
              <div className="summary-value">{totalUnits}</div>
            </div>
            <div className="summary-card">
              <span className="summary-icon"><i className="fas fa-money-bill"></i></span>
              <div>Total Amount</div>
              <div className="summary-value"> {totalAmount.toLocaleString()}</div>
            </div>
            <div className="summary-card">
              <span className="summary-icon"><i className="fas fa-check-circle"></i></span>
              <div>Paid Amount</div>
              <div className="summary-value"> {paidAmount.toLocaleString()}</div>
            </div>
            <div className="summary-card">
              <span className="summary-icon"><i className="fas fa-percent"></i></span>
              <div>Completion</div>
              <div className="summary-value">{completion}%</div>
            </div>
          </div>
        </div>
        <div className="summary-image">
          <img src="/MIPIMG.jpg" alt="Property" />
        </div>
      </div>

      {/* Units Table */}
      {units.length === 0 ? (
        <p>No units found.</p>
      ) : (
        <div className="units-table">
          <div className="units-header">
            <div className="units-col units-name">Unit</div>
            <div className="units-col units-project">Project</div>
            <div className="units-col units-price">Total Price</div>
            <div className="units-col units-paid">Paid Amount</div>
            <div className="units-col units-remaining">Remaining</div>
            <div className="units-col units-completion">Completion</div>
            <div className="units-col units-actions">Actions</div>
          </div>
          {units.map((unit, idx) => {
            const unitInstallments = installmentsMap[idx] || [];
            const unitTotalPrice = Number(unit.soldprice);
            const unitPaidAmount = unitInstallments
              .filter(inst => inst.status === "Paid")
              .reduce((sum, inst) => sum + Number(inst.amount), 0);
            const unitRemainingAmount = unitTotalPrice - unitPaidAmount;
            const unitCompletion = unitTotalPrice > 0 ? 
              Math.round((unitPaidAmount / unitTotalPrice) * 100) : 0;

            return (
              <React.Fragment key={idx}>
                <div className={`units-row${expandedIdx === idx ? " selected-unit-row" : ""}`}>
                  <div className="units-col units-name">
                    <div className="unit-main">{unit.unit}</div>
                  </div>

                  <div className="units-col units-project">
                   <div className="unit-project">{unit.project || '—'}</div>
                 </div>

                  <div className="units-col units-price">
                     {unitTotalPrice.toLocaleString()}
                  </div>
                  <div className="units-col units-paid">
                     {unitPaidAmount.toLocaleString()}
                  </div>
                  <div className="units-col units-remaining">
                     {unitRemainingAmount.toLocaleString()}
                  </div>
                  <div className="units-col units-completion">
                    <div className="completion-indicator">
                      <div className="completion-bar" style={{width: `${unitCompletion}%`}}></div>
                      <span>{unitCompletion}%</span>
                    </div>
                  </div>
                  <div className="units-col units-actions">
                    <button
                      className="installments-btn"
                      onClick={() => handleViewInstallments(unit, idx)}
                    >
                      Installments
                    </button>
                  </div>
                </div>

                {expandedIdx === idx && (
  <div className="installments-row">
    {/* Tabs Section */}
    <div className="installments-tabs">
      {["Paid", "Unpaid", "Overdue","Maintenance Differences"].map((tab) => (
        <button
          key={tab}
          className={`tab-btn ${activeTab === tab ? "active-tab" : ""}`}
          onClick={() => setActiveTab(tab)}
        >
          {tab}
        </button>
      ))}
    </div>

    {/* Installments Table */}
    <table className="installments-table">
      <thead>
        <tr>
          <th>Due Date</th>
          <th>Amount</th>
          <th>Type</th>
          <th>Status</th>
          <th>Actions</th>
        </tr>
      </thead>
      <tbody>
        {unitInstallments
          .filter((inst) => {
            const dueDate = new Date(inst.dueDate.replace(/\s/g, '-'));
            const isOverdue = inst.status === "Unpaid" && dueDate < new Date();

            if (activeTab === "Paid") return inst.status === "Paid";
            if (activeTab === "Unpaid") return inst.status === "Unpaid" && !isOverdue;
            if (activeTab === "Overdue") return isOverdue;
            if (activeTab === "Maintenance Differences") return inst.type === "Maintenance Difference";
            return true;
          })
          .map((inst, i) => {
            const dueDate = new Date(inst.dueDate.replace(/\s/g, '-'));
            const isOverdue = inst.status === "Unpaid" && dueDate < new Date();

            return (
              <tr key={i} className={isOverdue ? 'overdue-row' : ''}>
                <td>
                  {isOverdue && <span className="overdue-flag">!</span>}
                  <span className={isOverdue ? 'overdue-date' : ''}>
                    {inst.dueDate.replace(/\s/g, '/')}
                  </span>
                </td>
                <td>EGP {Number(inst.amount).toLocaleString()}</td>
                <td>{inst.type}</td>
                <td>
                  <span
                    className={`status-badge ${
                      inst.status === "Paid" ? "status-paid" : "status-unpaid"
                    }`}
                  >
                    {isOverdue ? "Overdue" : inst.status}
                  </span>
                </td>
                <td>
                  {inst.status === "Unpaid" ? (
                    <button
                      className={`pay-btn ${isOverdue ? 'pay-btn-urgent' : ''}`}
                      onClick={() => handlePayNow(inst, unit)}
                    >
                      Pay Now
                    </button>
                  ) : (
                    <span className="action-completed">Completed</span>
                  )}
                </td>
              </tr>
            );
          })}
      </tbody>
    </table>
  </div>
)}




              </React.Fragment>
            );
          })}
        </div>
      )}
    </div>
  );
};

export default Dashboard;
